'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _typeof = typeof Symbol === "function" && typeof Symbol.iterator === "symbol" ? function (obj) { return typeof obj; } : function (obj) { return obj && typeof Symbol === "function" && obj.constructor === Symbol && obj !== Symbol.prototype ? "symbol" : typeof obj; };

exports.default = convert;

var _assert = require('assert');

var _assert2 = _interopRequireDefault(_assert);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _toConsumableArray(arr) { if (Array.isArray(arr)) { for (var i = 0, arr2 = Array(arr.length); i < arr.length; i++) { arr2[i] = arr[i]; } return arr2; } else { return Array.from(arr); } }

// Converter helpers for Joi types.

var TYPES = {

  alternatives: function alternatives(schema, joi, transformer) {

    var result = schema.oneOf = [];

    joi._inner.matches.forEach(function (match) {

      if (match.schema) {
        return result.push(convert(match.schema, transformer));
      }

      if (!match.is) {
        throw new Error('joi.when requires an "is"');
      }
      if (!(match.then || match.otherwise)) {
        throw new Error('joi.when requires one or both of "then" and "otherwise"');
      }

      if (match.then) {
        result.push(convert(match.then, transformer));
      }

      if (match.otherwise) {
        result.push(convert(match.otherwise, transformer));
      }
    });
    return schema;
  },

  date: function date(schema, joi) {
    if (joi._flags.timestamp) {
      schema.type = 'integer';
      return schema;
    }

    schema.type = 'string';
    schema.format = 'date-time';
    return schema;
  },

  any: function any(schema) {
    schema.type = ["array", "boolean", 'number', "object", 'string', "null"];
    return schema;
  },

  array: function array(schema, joi, transformer) {
    schema.type = 'array';

    joi._tests.forEach(function (test) {
      switch (test.name) {
        case 'unique':
          schema.uniqueItems = true;
          break;
        case 'length':
          schema.minItems = schema.maxItems = test.arg;
          break;
        case 'min':
          schema.minItems = test.arg;
          break;
        case 'max':
          schema.maxItems = test.arg;
          break;
      }
    });

    if (joi._inner) {
      if (joi._inner.ordereds.length) {
        schema.ordered = joi._inner.ordereds.map(function (item) {
          return convert(item, transformer);
        });
      }

      var list = void 0;
      if (joi._inner.inclusions.length) {
        list = joi._inner.inclusions;
      } else if (joi._inner.requireds.length) {
        list = joi._inner.requireds;
      }

      if (list) {
        schema.items = convert(list[0], transformer);
      }
    }

    return schema;
  },

  binary: function binary(schema, joi) {
    schema.type = 'string';
    schema.contentMediaType = joi._meta.length > 0 && joi._meta[0].contentMediaType ? joi._meta[0].contentMediaType : 'text/plain';
    schema.contentEncoding = joi._flags.encoding ? joi._flags.encoding : 'binary';
    return schema;
  },

  boolean: function boolean(schema) {
    schema.type = 'boolean';
    return schema;
  },

  number: function number(schema, joi) {
    schema.type = 'number';
    joi._tests.forEach(function (test) {
      switch (test.name) {
        case 'integer':
          schema.type = 'integer';
          break;
        case 'less':
          schema.exclusiveMaximum = true;
          schema.maximum = test.arg;
          break;
        case 'greater':
          schema.exclusiveMinimum = true;
          schema.minimum = test.arg;
          break;
        case 'min':
          schema.minimum = test.arg;
          break;
        case 'max':
          schema.maximum = test.arg;
          break;
        case 'precision':
          var multipleOf = void 0;
          if (test.arg > 1) {
            multipleOf = JSON.parse('0.' + '0'.repeat(test.arg - 1) + '1');
          } else {
            multipleOf = 1;
          }
          schema.multipleOf = multipleOf;
          break;
      }
    });
    return schema;
  },

  string: function string(schema, joi) {
    schema.type = 'string';

    joi._tests.forEach(function (test) {
      switch (test.name) {
        case 'email':
          schema.format = 'email';
          break;
        case 'regex':
          // for backward compatibility
          var arg = test.arg;

          // This is required for backward compatibility
          // Location "pattern" had changed since Joi v9.0.0
          //
          // For example:
          //
          // before Joi v9: test.arg
          // since Joi v9: test.arg.pattern

          var pattern = arg && arg.pattern ? arg.pattern : arg;
          schema.pattern = String(pattern).replace(/^\//, '').replace(/\/$/, '');
          break;
        case 'min':
          schema.minLength = test.arg;
          break;
        case 'max':
          schema.maxLength = test.arg;
          break;
        case 'length':
          schema.minLength = schema.maxLength = test.arg;
          break;
        case 'uri':
          schema.format = 'uri';
          break;
      }
    });

    return schema;
  },

  object: function object(schema, joi, transformer) {
    schema.type = 'object';
    schema.properties = {};
    schema.additionalProperties = Boolean(joi._flags.allowUnknown);
    schema.patterns = joi._inner.patterns.map(function (pattern) {
      return { regex: pattern.regex, rule: convert(pattern.rule, transformer) };
    });

    if (!joi._inner.children) {
      return schema;
    }

    joi._inner.children.forEach(function (property) {
      if (property.schema._flags.presence !== 'forbidden') {
        schema.properties[property.key] = convert(property.schema, transformer);
        if (property.schema._flags.presence === 'required') {
          schema.required = schema.required || [];
          schema.required.push(property.key);
        }
      }
    });

    return schema;
  }
};

/**
 * Converts the supplied joi validation object into a JSON schema object,
 * optionally applying a transformation.
 *
 * @param {JoiValidation} joi
 * @param {TransformFunction} [transformer=null]
 * @returns {JSONSchema}
 */
function convert(joi) {
  var transformer = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : null;


  (0, _assert2.default)('object' === (typeof joi === 'undefined' ? 'undefined' : _typeof(joi)) && true === joi.isJoi, 'requires a joi schema object');

  (0, _assert2.default)(joi._type, 'joi schema object must have a type');

  if (!TYPES[joi._type]) {
    throw new Error('sorry, do not know how to convert unknown joi type: "' + joi._type + '"');
  }

  if (transformer) {
    (0, _assert2.default)('function' === typeof transformer, 'transformer must be a function');
  }

  // JSON Schema root for this type.
  var schema = {};

  // Copy over the details that all schemas may have...
  if (joi._description) {
    schema.description = joi._description;
  }

  if (joi._examples && joi._examples.length > 0) {
    schema.examples = joi._examples;
  }

  if (joi._examples && joi._examples.length === 1) {
    schema.example = joi._examples[0];
  }

  // Add the label as a title if it exists
  if (joi._settings && joi._settings.language && joi._settings.language.label) {
    schema.title = joi._settings.language.label;
  } else if (joi._flags && joi._flags.label) {
    schema.title = joi._flags.label;
  }

  // Checking for undefined and null explicitly to allow false and 0 values
  if (joi._flags && joi._flags.default !== undefined && joi._flags.default !== null) {
    schema['default'] = joi._flags.default;
  }

  if (joi._valids && joi._valids._set && (joi._valids._set.size || joi._valids._set.length)) {
    if (Array.isArray(joi._inner.children) || !joi._flags.allowOnly) {
      return {
        'anyOf': [{
          'type': joi._type,
          'enum': [].concat(_toConsumableArray(joi._valids._set))
        }, TYPES[joi._type](schema, joi, transformer)]
      };
    }
    schema['enum'] = [].concat(_toConsumableArray(joi._valids._set));
  }

  var result = TYPES[joi._type](schema, joi, transformer);

  if (transformer) {
    result = transformer(result, joi);
  }

  return result;
}

module.exports = exports = convert;
convert.TYPES = TYPES;

/**
 * Joi Validation Object
 * @typedef {object} JoiValidation
 */

/**
 * Transformation Function - applied just before `convert()` returns and called as `function(object):object`
 * @typedef {function} TransformFunction
 */

/**
 * JSON Schema Object
 * @typedef {object} JSONSchema
 */
//# sourceMappingURL=index.js.map